/******************************************************************************
 *
 * Copyright (c) 2004 PalmSource, Inc. All rights reserved.
 *
 * File: SharedLib.c
 *
 * Description:
 *		This is the main source module for a Palm OS shared library.
 *		Based on fine examples from PRC-tools.
 *
 *****************************************************************************/

#include <SystemMgr.h>

#define SHARED_LIB_TRAP(trapNum)	/* Needs to be prior to SharedLib.h inclusion */

#include "SharedLib.h"


typedef struct {
	UInt16 libraryReferenceCount;
} SharedLib_globals;


/***********************************************************************
 *
 * FUNCTION:    start
 *
 * DESCRIPTION: A function all shared libraries must have to patch the
 * 				jump table to enable this shared library. 
 *
 ***********************************************************************/

UInt32 start(UInt16 libraryReference, SysLibTblEntryPtr entryP) {
	extern void *jmptable ();
	
	entryP->dispatchTblP = (void *)jmptable;
	entryP->globalsP = NULL;
	
	return errNone;
}

/***********************************************************************
 *
 * FUNCTION:    SharedLib_Open
 *
 * DESCRIPTION: A function that updates the reference count on how
 * 				many applications have the library open. 
 *
 ***********************************************************************/

UInt32 SharedLib_Open(UInt16 libraryReference) {
	SysLibTblEntryPtr entryP = SysLibTblEntry(libraryReference);
	SharedLib_globals* globals = entryP->globalsP;

	/* If no entry, create one */
	if (globals == NULL) {
		entryP->globalsP = MemPtrNew(sizeof(SharedLib_globals));
		globals = entryP->globalsP;
		MemPtrSetOwner(globals, 0);
		globals->libraryReferenceCount = 0;
	}

	/* Increment the references */
  	++globals->libraryReferenceCount;

	return errNone;
}

/***********************************************************************
 *
 * FUNCTION:    SharedLib_Close
 *
 * DESCRIPTION: A function that decrements the reference count on how
 * 				many applications have the library open. 
 *
 ***********************************************************************/

UInt32 SharedLib_Close(UInt16 libraryReference, UInt16* referencesRemaining) {
	SysLibTblEntryPtr entryP = SysLibTblEntry(libraryReference);
	SharedLib_globals *globals = entryP->globalsP;

	if (globals == NULL) {
		/* Something went wrong... */
		return 1;
    }
  
 	*referencesRemaining = globals->libraryReferenceCount - 1;
  
  	/* If no more references free memory */
	if (*referencesRemaining == 0) {
		MemChunkFree (entryP->globalsP);
		entryP->globalsP = NULL;
	}

	return errNone;
}

/***********************************************************************
 *
 * FUNCTION:    SharedLib_Sleep
 *
 * DESCRIPTION: A function needed for handling sleep. 
 *
 ***********************************************************************/

UInt32 SharedLib_Sleep(UInt16 libraryReference)
{
	return errNone;
}
	
/***********************************************************************
 *
 * FUNCTION:    SharedLib_Wake
 *
 * DESCRIPTION: A function needed for handling wake. 
 *
 ***********************************************************************/

UInt32 SharedLib_Wake(UInt16 libraryReference)
{
	return errNone;
}


/***********************************************************************
 *
 * FUNCTION:    SharedLib_FunctionA
 *
 * DESCRIPTION: First sample library function that returns the parameter
 * provided. 
 *
 ***********************************************************************/
 
void SharedLib_FunctionA(UInt16 libraryReference, Int32 parameter, Int32* result) {
	*result = parameter;
}

/***********************************************************************
 *
 * FUNCTION:    SharedLib_FunctionB
 *
 * DESCRIPTION: Second sample library function that does the same as A.
 *
 ***********************************************************************/
 
void SharedLib_FunctionB(UInt16 libraryReference, Int32 parameter, Int32* result) {
	*result = parameter;
}


